using System.Collections.Generic;
using UnityEngine;

/*
 * Handler for the Indvidual Voter Agents used in the Visualised LD Simualtions
 */
public class VoterHandler : MonoBehaviour
{
    public int voterID;
    double competence = 0.5;
    public double percievedCompetence = 0.5;
    float delegationChance = 95;

    Dictionary<int, VoterHandler> localNetwork = new Dictionary<int, VoterHandler>();
    bool delegates = false;
    float alpha = 0.02f;
    VoterHandler delegatesTo;
    VoterHandler guru;
    VoterHandler prevGuru;
    DelegationHandler delegationLine;

    public int vote;
    bool reviewed;
    public int correctVotes;
    public int incorrectVotes;

    float preferentialAttachment;
    float competenceImportance;

    Dictionary<int, VoterHandler> delegators = new Dictionary<int, VoterHandler>();
    Dictionary<int, VoterHandler> directDelegators = new Dictionary<int, VoterHandler>();

    public GameObject voter;
    public GameObject head;
    public GameObject body;
    public GameObject hat;
    Color guruColour;
    Animator voterAnim;
    float speed;

    public bool moving;
    int movementCycle;
    int correctOption;
    Vector3 startPos;
    Vector3 moveTo;
    Vector3 entA;
    Vector3 exA;
    Vector3 entB;
    Vector3 exB;

    LiquidDemocracy env;

    // Start is called before the first frame update, sets default values for visualised voter
    void Start()
    {
        guru = this;
        prevGuru = this;
        guruColour = new Color((float)Random.Range(0, 100) / 100, (float)Random.Range(0, 100) / 100, (float)Random.Range(0, 100) / 100, 1);
        hat.GetComponent<Renderer>().material.color = guruColour;
        voterAnim = voter.GetComponent<Animator>();
        movementCycle = 0;
        correctVotes = 0;
        incorrectVotes = 0;
        startPos = transform.position;
    }

    // Deals with voter movement when voting and returning to start position
    void FixedUpdate()
    {
        voterAnim.SetFloat("Speed", speed);

        if (moving)
        {
            Vector3 target = moveTo;

            switch (movementCycle)
            {
                case 0:
                    if (vote == 0 && correctOption == 1) { target = entA; }
                    else if (vote == 1 && correctOption == 0) { target = entA; }
                    else if (vote == 0 && correctOption == 0) { target = entB; }
                    else { target = entB; }
                    break;

                case 1:
                    if (vote == 0 && correctOption == 1) { target = exA; }
                    else if (vote == 1 && correctOption == 0) { target = exA; }
                    else if (vote == 0 && correctOption == 0) { target = exB; }
                    else { target = exB; }
                    break;

                case 2:
                    target = moveTo;
                    break;

                case 4:
                    if (vote == 0 && correctOption == 1) { target = exA; }
                    else if (vote == 1 && correctOption == 0) { target = exA; }
                    else if (vote == 0 && correctOption == 0) { target = exB; }
                    else { target = exB; }
                    break;

                case 5:
                    if (vote == 0 && correctOption == 1) { target = entA; }
                    else if (vote == 1 && correctOption == 0) { target = entA; }
                    else if (vote == 0 && correctOption == 0) { target = entB; }
                    else { target = entB; }
                    break;

                case 6:
                    target = startPos;
                    break;
            }

            transform.position = Vector3.MoveTowards(transform.position, target, 0.12f * speed);
           
            if (transform.position == target) { movementCycle++; }

            if (transform.position == entA) { body.GetComponent<Renderer>().material.color = new Color(0.3f, 1, 0.9f, 1); }
            else if (transform.position == entB) { body.GetComponent<Renderer>().material.color = new Color(1, 0.45f, 0.8f, 1); }

            if (movementCycle == 3 || movementCycle == 7)
            {
                moving = false;
                voterAnim.SetTrigger("Stop");
            }

            if ((transform.position - target).sqrMagnitude >= 0.001f)
            {
                Quaternion targetRotation = Quaternion.LookRotation(transform.position - target);
                transform.rotation = Quaternion.Lerp(transform.rotation, targetRotation, 3 * Time.deltaTime);
            }
        }
        else
        {
            Vector3 look = new Vector3(transform.position.x, transform.position.y, transform.position.z -1);

            if ((transform.position - look).sqrMagnitude >= 0.001f)
            { 
                Quaternion targetRotation = Quaternion.LookRotation(transform.position - look);
                transform.rotation = Quaternion.Lerp(transform.rotation, targetRotation, 3 * Time.deltaTime);
            }
        }

    }

    // Sets up voter agent, both parameters and visualised components, with user settings and generated numbers
    public void SetupVoter(int voterID, double addCompetence, float speed, float prefAttach, float compImp, float alphaValue, LiquidDemocracy env)
    {
        this.voterID = voterID;
        this.env = env;

        competence += (addCompetence / 100);
        if (competence > 0.95f) { competence = 0.95f; }

        if (StatsManager.instance.roundType == 0)
        {
            percievedCompetence = competence;
        }
        else
        {
            percievedCompetence = 0.5;
        }


        this.speed = speed;
        preferentialAttachment = prefAttach;
        competenceImportance = compImp;
        alpha = alphaValue;

        hat.SetActive(false);

        // Sets brightness of voter head based on perceived competence
        double brightness = ((percievedCompetence - 0.5f) / 0.25f) * 0.510f;
        if (brightness > 0.510f) { brightness = 0.510f; }
        head.GetComponent<Renderer>().material.color = new Color(0.863f, 0.824f, (float)(0.510f - brightness), 1);

        delegationLine = gameObject.AddComponent<DelegationHandler>();
        
        // Calculates chance of delegation
        if (!StatsManager.instance.setChance)
        {
            if (competence * 100 >= 51)
            {
                delegationChance = 95 - ((float)System.Math.Log((competence * 100) - 50) / (float)System.Math.Log(45) * 90);
            }
        }
        else
        {
            delegationChance = 50;
        }
    }

    // Local network of voters the voter knows added into dictionary
    public void SetupNetwork(int networkSize, VoterHandler[] voterNetwork)
    {
        int netStart;
        int netEnd;

        if (networkSize % 2 == 0)
        {
            if (Random.Range(0, 2) == 0)
            {
                netStart = voterID - networkSize / 2;
                netEnd = voterID + (networkSize / 2 - 1);
            }
            else
            {
                netStart = voterID - (networkSize / 2 - 1);
                netEnd = voterID + (networkSize / 2);
            }
        }
        else
        {
            netStart = voterID - (networkSize - 1) / 2;
            netEnd = voterID + (networkSize - 1) / 2;
        }
        if (netStart < 0)
        {
            netEnd += Mathf.Abs(netStart);
            netStart = 0;
        }
        else if (netEnd >= StatsManager.instance.numVoters)
        {
            netStart -= (netEnd + 1 - StatsManager.instance.numVoters);
            netEnd = StatsManager.instance.numVoters - 1;
        }

        for (int i = netStart; i <= netEnd; i++)
        {
            if (voterNetwork[i].voterID != voterID)
            {
                localNetwork.Add(voterNetwork[i].voterID, voterNetwork[i]);
            }
        }
    }

    // Correct option given FOR MOVEMENT VISUALISATION ONLY, the voter would not actually be aware of this
    public void NewVote(int correctOption)
    {
        this.correctOption = correctOption;
    }


    // Voter decides whether to be a guru or delegator based on delegation chance
    public void GuruOrDelegator(VoterHandler[] voterNetwork)
    {

        int isGuru;
        int guruChance = Random.Range(1, 101); 
        if (guruChance > delegationChance) { isGuru = 1; }
        else { isGuru = 0; }

        // Chooses to be a Delegator
        if (isGuru == 0)
        {
            delegates = true;

            int numHigherComp = 0;
            while (voterNetwork[numHigherComp].percievedCompetence >= competence + alpha)
            {
                numHigherComp++;
            }

            if (percievedCompetence > competence + alpha)
            {
                numHigherComp--;
            }

            int numAccDel = 0;
            for(int i = 0; i < numHigherComp; i++)
            {
                if(localNetwork.ContainsKey(voterNetwork[i].voterID) 
                    && voterNetwork[i].guru.voterID != prevGuru.voterID && !delegators.ContainsKey(voterNetwork[i].voterID))
                {
                    numAccDel++;
                }
            }

            // If there are no accetapble delegations voter will have to be a Guru, else list of acceptable options made
            if (numAccDel == 0) { isGuru = 1; }
            else
            {
                VoterHandler[] accDelegations = new VoterHandler[numAccDel];
                int i = 0;
                for (int j = 0; j < numHigherComp; j++)
                {
                    if (localNetwork.ContainsKey(voterNetwork[j].voterID)
                        && voterNetwork[j].guru.voterID != prevGuru.voterID && !delegators.ContainsKey(voterNetwork[j].voterID))
                    {
                        accDelegations[i] = voterNetwork[j];
                        i++;
                        if(i >= numAccDel) { break; }
                    }
                }

                hat.SetActive(false);
                GiveDelegation(numAccDel, accDelegations);
            }
        }

        // Chooses (or has) to be a Guru
        if(isGuru == 1)
        {
            delegates = false;
            hat.SetActive(true);
        }
    }

    // Voter delegates their vote to another voter
    public void GiveDelegation(int numAccDel, VoterHandler[] accDelegations)
    {
        // Chance of picking each delegation option calculated
        int totalWeight = 0;
        int[] powerWeights = new int[numAccDel];
        for (int i = 0; i < numAccDel; i++)
        {
            int prefPower = (int)Mathf.Pow(1 + accDelegations[i].delegators.Count, preferentialAttachment);
            int compPower = (int)Mathf.Pow(1 + (100 * ((float)accDelegations[i].percievedCompetence - (float)competence)), competenceImportance);

            int power = prefPower + compPower;
            powerWeights[i] = totalWeight + power;

            totalWeight += power;
        }

        // Option selected
        int delegation = 0;
        int randomChoice = Random.Range(0, totalWeight);
        foreach (int powerWeight in powerWeights)
        {
            if (randomChoice <= powerWeight)
            {
                break;
            }
            delegation++;
        }

        delegatesTo = accDelegations[delegation];
        delegationLine.SetDelegation(this, delegatesTo);

        accDelegations[delegation].RecieveDelegation(this, true);
    }

    // Voter removes their current delegation
    public void RemoveDelegation()
    {
        delegates = false;
        delegationLine.StopDelegation();
        prevGuru = guru;
        guru = this;

        // Delegate loses the delegation from the voter
        delegatesTo.LoseDelegation(this, true);
        delegatesTo = null;
    }

    // Voter receives a delegation from another voter
    public void RecieveDelegation(VoterHandler delegator, bool direct)
    {
        if (!directDelegators.ContainsKey(delegator.voterID) && direct)
        {
            directDelegators.Add(delegator.voterID, delegator);
        }

        if (!delegators.ContainsKey(delegator.voterID))
        {
            delegators.Add(delegator.voterID, delegator);
            delegator.SetGuru(this);
            body.transform.localScale += new Vector3(0, 0.2f, 0);
            transform.position += new Vector3(0, 0.2f, 0);
            body.transform.position += new Vector3(0, -0.1f, 0);
        }

        // Delegators of delegating voter added as delegators to this voter
        foreach(VoterHandler subDelegator in delegator.delegators.Values)
        {
            if (!delegators.ContainsKey(subDelegator.voterID))
            {
                //print("Voter " + voterID + " recieves sub-delegate " + subDelegator.voterID);

                delegators.Add(subDelegator.voterID, subDelegator);
                subDelegator.SetGuru(this);
                body.transform.localScale += new Vector3(0, 0.2f, 0);
                transform.position += new Vector3(0, 0.2f, 0);
                body.transform.position += new Vector3(0, -0.1f, 0);
            }
        }

        // If this voter is a delegator too it passes on the delegations down the transitive delegation chain
        if (delegates)
        {
            delegatesTo.RecieveDelegation(delegator, false);
        }
    }

    // Voter loses a delegation from another voter
    public void LoseDelegation(VoterHandler delegator, bool direct)
    {
        if (direct)
        {
            directDelegators.Remove(delegator.voterID);
        }

        delegators.Remove(delegator.voterID);

        body.transform.localScale -= new Vector3(0, 0.2f, 0);
        transform.position -= new Vector3(0, 0.2f, 0);
        body.transform.position -= new Vector3(0, -0.1f, 0);

        // Also loses the sub delegators of the delegator
        foreach (VoterHandler subDelegator in delegator.delegators.Values)
        {
            if (delegators.ContainsKey(subDelegator.voterID))
            {
                delegators.Remove(subDelegator.voterID);
                subDelegator.SetGuru(delegator);
                subDelegator.prevGuru = this;

                body.transform.localScale -= new Vector3(0, 0.2f, 0);
                transform.position -= new Vector3(0, 0.2f, 0);
                body.transform.position -= new Vector3(0, -0.1f, 0);
            }
        }

        // If this voter is a delegator the voter it delegates to loses the delegations too
        if (delegates)
        {
            delegatesTo.LoseDelegation(delegator, false);
        }
    }

    // Voter votes, makes right decision based on competence, only used by Gurus
    public void CastVote(out int vote, out int power)
    {
        int decision = Random.Range(1, 101);
        if(competence * 100 >= decision) 
        { 
            vote = 1;
            correctVotes++;
        }
        else 
        { 
            vote = 0;
            incorrectVotes++;
        }
        this.vote = vote;
        reviewed = false;

        power = 1 + delegators.Values.Count;

        // Starts voter moving to chosen vote option area
        SetPaths();
        MoveToVote();

        // Delegators of voter receive vote option, move to area with their guru
        foreach(VoterHandler delegator in delegators.Values)
        {
            delegator.GuruVote(vote);
        }
    }

    // Recieve vote option made by guru
    public void GuruVote(int vote)
    {
        SetPaths();
        this.vote = vote;
        reviewed = false;
        MoveToVote();
    }

    // Returns if voter is a guru
    public bool IsGuru()
    {
        return !delegates;
    }

    // Return competence of voter
    public double GetTrueCompetence()
    {
        return competence;
    }

    // Return power of voter (1 + number of delegators)
    public int GetPower()
    {
        if (!delegates) {  return 1 + delegators.Values.Count; }
        else { return 0; }
    }

    // Gets ID of Voter Delegate (if voter does not delegate it will be the voter's ID)
    public int GetDelegateID()
    {
        if (delegates) { return delegatesTo.voterID;  }
        else { return voterID; }
    }

    // Gets ID of Voter Guru (if voter does not delegate it will be the voter's ID)
    public int GetGuruID()
    {
        if (delegates) { return guru.voterID; }
        else { return voterID; }
    }

    // Returns string list of voter's delegators
    public string ListDelegators()
    {
        if (delegators.Count > 0)
        {
            string delegatorList = "";
            foreach (int delegatorID in delegators.Keys)
            {
                delegatorList = delegatorList + delegatorID + ", ";
            }

            return delegatorList.Substring(0, delegatorList.Length - 2);
        }
        else
        {
            return "None";
        }
    }

    // Returns string list of voter's direct delegators
    public string ListDirectDelegators()
    {
        if (directDelegators.Count > 0)
        {
            string delegatorList = "";
            foreach (int delegatorID in directDelegators.Keys)
            {
                delegatorList = delegatorList + delegatorID + ", ";
            }

            return delegatorList.Substring(0, delegatorList.Length - 2);
        }
        else
        {
            return "None";
        }
    }

    // Sets voter path for moving to vote option area
    void SetPaths()
    {
        entA = new Vector3(-17, transform.position.y, 10);
        exA = new Vector3(-17, transform.position.y, 24);
        entB = new Vector3(17, transform.position.y, 10);
        exB = new Vector3(17, transform.position.y, 24);
    }

    // Starts voter moving to vote option area
    void MoveToVote()
    {
        startPos = transform.position;
        movementCycle = 0;

        int zPos = Random.Range(26, 48);
        int xPos;
        if (vote == 0 && correctOption == 1) { xPos = Random.Range(-29, -5); }
        else if (vote == 1 && correctOption == 0) { xPos = Random.Range(-29, -5); }
        else if (vote == 0 && correctOption == 0) { xPos = Random.Range(4, 28); }
        else { xPos = Random.Range(4, 28); }

        moveTo = new Vector3(xPos, transform.position.y, zPos);
        moving = true;
        voterAnim.SetTrigger("Move");
    }

    // Animates voter happy when their option wins or the community makes a correct decision
    public void Win()
    {
        voterAnim.SetTrigger("Win");
    }

    // Animates voter sad when their option loses or the community makes an incorrect decision
    public void Lose()
    {
        voterAnim.SetTrigger("Lose");
    }

    // Perceived competence of voter updated based on how many correct votes they have made
    public void UpdatePercievedComp()
    {
        percievedCompetence = (double)correctVotes / (correctVotes + incorrectVotes);
        if (percievedCompetence < 0.5) { percievedCompetence = 0.5; }

        double brightness = ((percievedCompetence - 0.5f) / 0.25f) * 0.510f;
        if (brightness > 0.510f) { brightness = 0.510f; }
        head.GetComponent<Renderer>().material.color = new Color(0.863f, 0.824f, (float)(0.510f - brightness), 1);
    }


    // Guru of voter is set
    public void SetGuru(VoterHandler guru)
    {
        this.guru = guru;
        delegationLine.setColour(guru.guruColour);

        // Add guru to local network if voter was not already aware of it
        if (!localNetwork.ContainsKey(guru.voterID))
        {
            localNetwork.Add(guru.voterID, guru);
        }
    }

    // Makes voter return to their start position in the environment
    public void ReturnToStart()
    {
        movementCycle = 4;
        moving = true;
        voterAnim.SetTrigger("Move");
    }

    // Voter checks if they made the right decision if they were a guru last round
    public void ReviewVote(VoterHandler[] votersByComp)
    {
        if (!reviewed)
        {
            reviewed = true;

            // If wrong they again make the guru or delegator decision, else they remain a guru
            if (vote == 0)
            {
                VoterHandler[] potentialyLostDelegates = new VoterHandler[directDelegators.Count];
                directDelegators.Values.CopyTo(potentialyLostDelegates, 0);

                foreach (VoterHandler voter in potentialyLostDelegates)
                {
                    voter.ReviewDelegation(votersByComp);
                }

                GuruOrDelegator(votersByComp);
            }
        }
    }

    // Voter checks if their guru made the right decision if they were a delegator last round
    public void ReviewDelegation(VoterHandler[] votersByComp)
    {
        if (!reviewed)
        {
            reviewed = true;

            // Voter removes delegation with chance based on their guru's competence if their guru was wrong last round
            if (vote == 0)
            {
                double changeChance = 50 - 100 * (guru.percievedCompetence - competence);
                if(guru.percievedCompetence < competence)
                {
                    changeChance = 100;
                }
                int changeChoice = Random.Range(1, 101);
                if (changeChance >= changeChoice)
                {
                    RemoveDelegation();
                    GuruOrDelegator(votersByComp);
                }
                else if(directDelegators.Count > 0)
                {
                    VoterHandler[] potentialyLostDelegates = new VoterHandler[directDelegators.Count];
                    directDelegators.Values.CopyTo(potentialyLostDelegates, 0);

                    // Each sub delegator who does not directly delegate to the wrong guru has a chance to remove their delegation if their delegate does not
                    foreach (VoterHandler voter in potentialyLostDelegates)
                    {
                        voter.ReviewDelegation(votersByComp);
                    }
                }
            }
        }
    }

    // Voter Agent is clicked on, so their stats are set to be displayed by the environment's agent info UI
    void OnMouseDown()
    {
        env.DisplayAgentStats(this);
    }

    // Voter Agent removed from simulation
    public void Remove()
    {
        if (delegates) { delegationLine.Remove(); };
        Destroy(gameObject);
        Destroy(this);
    }
}